//=============================================================================
// ElementThrethold.js
// ----------------------------------------------------------------------------
// (C)2024 kuroame koubou
// This software is released under the MIT License.
// http://opensource.org/licenses/mit-license.php
// ----------------------------------------------------------------------------
// Version
// 0.9.0 2024/12/14 初版
// ----------------------------------------------------------------------------
// [Blog]   : http://kuroamekoubou.blog.fc2.com
// [Twitter]: https://twitter.com/kuroVelter9623
//=============================================================================

/*:
 * @plugindesc 属性倍率閾値プラグイン
 * @target MZ
 * @author くろあめ
 *
 * @help
 * 指定のメタタグをメモ欄に記載することで
 * その要素を持つアクター/エネミーが攻撃した際に
 * 属性耐性でダメージを軽減された/弱点属性でダメージを増加させたとき
 * その倍率を変動させる機能を搭載するプラグインです。
 * 
 * アクター、エネミー、スキル、職業、装備、ステートのメモ欄に
 * <ElmResistPass>と記載することで、その要素を持つアクター/エネミーが攻撃した際に
 * 最低の属性倍率100%を保証します（属性軽減を無かったことにします）
 *
 *
 * アクター、エネミー、スキル、職業、装備、ステートのメモ欄に
 * <WeakGuarantee:XXX>と記載することで、その要素を持つアクター/エネミーが攻撃した際に
 * パラメータで指定した最低保証属性倍率を保証します。
 * （属性倍率が指定倍率より低かった場合、指定倍率まで引き上げます）
 * 
 * 複数記述がある場合は、最大値を採用します。
 *
 * 【設定例】
 * <WeakGuarantee:1.2> とメモ欄に記載されたアクセサリを装備したアクターが攻撃する際に
 * 属性倍率が120%を下回る場合、最終的な属性倍率が120%に置き換わります。
 *
 *
 * アクター、エネミー、スキル、職業、装備、ステートのメモ欄に
 * <WeakAmplify:xxx>と記載することで、その要素を持つアクター/エネミーが攻撃した際に
 * 属性倍率が1.0より高ければ、指定した倍率を属性倍率に乗算します。
 * （属性倍率が1.0より低かった場合、特に何も起きません） 
 * 
 * 【設定例】
 * <WeakAmplify:2.0> とメモ欄に記載されたスキルで単体対象に攻撃し、元々の属性倍率が150% の場合
 * 最終的な属性倍率は300%になります。
 *
 *
 * アクター、エネミー、スキル、職業、装備、ステートのメモ欄に
 * <ResistReverse>と記載することで、その要素を持つアクター/エネミーが攻撃した際に
 * 属性倍率が100%を下回っていた場合、以下の計算式で倍率を反転させます
 *
 * 反転倍率 = 1.0 + (1.0 - 属性軽減値)
 * 例：本来属性倍率85％で計算されるべき攻撃行動に、<ResistReverse>タグが記載されていた場合
 *     最終的な適用倍率は「115%」となる ⇒ 1.0 + (1.0 - 0.85) =  1.15
 *
 *
 * このプラグインのタグは、実行するスキルのダメージタイプが
 * 「HPダメージ」「HP吸収」のいずれかの場合のみ、適用されます
 * 回復スキルやMPダメージ/吸収には適用されません。
 *
 * 利用形態（商用、18禁利用等）について制限はありません
 * 作者への使用報告や、ReadMeなどへ作者・サイト名の記載も任意となります。
 *
 * ただし、作者はこれらのプラグインについて
 * 基本的に一切のサポートを行いません
 * 何か問題が起きた時、ご自身で解決できる方のみご利用ください。
 * 
 * @param WeakGuarantee Rate
 * @text 最低保証倍率
 * @desc <WeakGuarantee>タグ適用時に、最低保証する属性倍率です。1.0以上で確定弱点判定となります。
 * @default 1.1
 * @type number
 *
 */


(function() {

const parameters = PluginManager.parameters('ElementThrethold');
const weakGuaranteeRate = Number(parameters['WeakGuarantee Rate']);

/* 属性閾値判定 */

Game_BattlerBase.prototype.elementThretholdCheck = function(target,value,item) {

    if(value > 1.00){
        return value;
    }
    
    if (item && item.meta && item.meta.ElmResistPass) {
        return 1.00;
    }
    
    const states = this.states();
    for (let i = 0; i < states.length; i++) {
        if (states[i] && states[i].meta && states[i].meta.ElmResistPass){
            return 1.00;
        }
    }

    if(this.isActor()){
        if (this.actor() && this.actor().meta && this.actor().meta.ElmResistPass) {
            return 1.00;
        }

        if (this.currentClass() && this.currentClass().meta && this.currentClass().meta.ElmResistPass)  {
            return 1.00;
        }
        
        const equips = this.equips();
        for (let i = 0; i < equips.length; i++) {      
            if (equips[i] && equips[i].meta && equips[i].meta.ElmResistPass){
                return 1.00;
            }
        }
    }
    
    else{
        if (this.enemy() && this.enemy().meta && this.enemy().meta.ElmResistPass) {
            return 1.00;
        }
    }
    return value;
};


Game_BattlerBase.prototype.weakGuaranteeCheck = function(target,value,item) {

    var guarantee;
    
    if (item && item.meta && item.meta.WeakGuarantee) {
        guarantee = Number(item.meta.WeakGuarantee);
        value = value > guarantee ? value : guarantee;
    }
    
    const states = this.states();
    for (let i = 0; i < states.length; i++) {
        if (states[i] && states[i].meta && states[i].meta.WeakGuarantee){
            guarantee = Number(states[i].meta.WeakGuarantee);
            value = value > guarantee ? value : guarantee;
        }
    }

    if(this.isActor()){
        if (this.actor() && this.actor().meta && this.actor().meta.WeakGuarantee) {
            guarantee = Number(this.actor().meta.WeakGuarantee);
            value = value > guarantee ? value : guarantee;
        }

        if (this.currentClass() && this.currentClass().meta && this.currentClass().meta.WeakGuarantee)  {
            guarantee = Number(this.currentClass().meta.WeakGuarantee);
            value = value > guarantee ? value : guarantee;
        }
        
        const equips = this.equips();
        for (let i = 0; i < equips.length; i++) {      
            if (equips[i] && equips[i].meta && equips[i].meta.WeakGuarantee){
                guarantee = Number(equips[i].meta.WeakGuarantee);
                value = value > guarantee ? value : guarantee;
            }
        }
    }
    
    else{
        if (this.enemy() && this.enemy().meta && this.enemy().meta.WeakGuarantee) {
            guarantee = Number(this.enemy().meta.WeakGuarantee);
            value = value > guarantee ? value : guarantee;
        }
    }
    
    return value;
};

Game_BattlerBase.prototype.weakAmplify = function(target,value,item) {

    var amplify;

    if(value > 1.00){
    
        if (item && item.meta && item.meta.WeakAmplify) {
            amplify = Number(item.meta.WeakAmplify);
            value = value * amplify / 100;
        }
        
        const states = this.states();
        for (let i = 0; i < states.length; i++) {
            if (states[i] && states[i].meta && states[i].meta.WeakAmplify){
                amplify  = Number(states[i].meta.WeakAmplify);
                value = value * amplify;
            }
        }

        if(this.isActor()){
            if (this.actor() && this.actor().meta && this.actor().meta.WeakAmplify) {
                amplify = Number(this.actor().meta.WeakAmplify);
                value = value * amplify;
            }

            if (this.currentClass() && this.currentClass().meta && this.currentClass().meta.WeakAmplify)  {
                amplify = Number(this.currentClass().meta.WeakAmplify);
                value = value * amplify;
            }
            
            const equips = this.equips();
            for (let i = 0; i < equips.length; i++) {      
                if (equips[i] && equips[i].meta && equips[i].meta.WeakAmplify){
                    amplify = Number(equips[i].meta.WeakAmplify);
                    value = value * amplify;
                }
            }
        }
        else{
            if (this.enemy() && this.enemy().meta && this.enemy().meta.WeakAmplify) {
                amplify  = Number(this.enemy().meta.WeakAmplify);
                value = value * amplify;
            }
        }
    }
    else if(value < 1.00){
        
        var resistReverse = false;
    
        if (item && item.meta && item.meta.ResistReverse) {
            resistReverse = true;
        }
        
        const states = this.states();
        for (let i = 0; i < states.length; i++) {
            if (states[i] && states[i].meta && states[i].meta.ResistReverse){
                resistReverse = true;
            }
        }

        if(this.isActor()){
            if (this.actor() && this.actor().meta && this.actor().meta.ResistReverse) {
                resistReverse = true;
            }

            if (this.currentClass() && this.currentClass().meta && this.currentClass().meta.ResistReverse)  {
                resistReverse = true;
            }
            
            const equips = this.equips();
            for (let i = 0; i < equips.length; i++) {      
                if (equips[i] && equips[i].meta && equips[i].meta.ResistReverse){
                    resistReverse = true;
                }
            }
        }
        else{
            if (this.enemy() && this.enemy().meta && this.enemy().meta.ResistReverse) {
                resistReverse = true;
            }
        }
        
        if(resistReverse){
            value = 1.0 + (1.0 - value);
        }
    }
    
    return value;
    
};

const _Game_Action_calcElementRate = Game_Action.prototype.calcElementRate;
Game_Action.prototype.calcElementRate = function(target) {
    var result = _Game_Action_calcElementRate.apply(this, arguments);
    
    //console.log(this.item().damage.type);
    
    if(this.item().damage.type == 1 || this.item().damage.type == 5){
         //console.log(this.subject().name());
         //console.log(this.item().name);
         
         //console.log("beforVal");
         //console.log(result);
         
         result = this.subject().elementThretholdCheck(target,result,this.item());
         result = this.subject().weakGuaranteeCheck(target,result,this.item());
         result = this.subject().weakAmplify(target,result,this.item());
         
         //console.log("afterVal");
         //console.log(result);
    }
    return result;
};

})();